//*************************************************************************************************
//
//	Description:
//		reflectiveTrack.fx - Shader to display reflected world + cars
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Matt Hobbs
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		MHobbs		    23/10/2007  0.01		  Created
//	<TABLE>
//
//*************************************************************************************************
//#define _3DSMAX_ /*testing*/
#define _SSAO_READY_
#include "stddefs.fxh"
#include "specialisation_globals.fxh"
#include "lighting_globals.fxh"



//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#ifndef _3DSMAX_

// testing:-
//#define SHOW_DIFFUSE_TEXTURES
//#define SHOW_DARKENING_CALC
//#define SHOW_TRACK_TEXTURE
//#define SHOW_CARS_TEXTURE
//#define SHOW_FRESNEL

// visual options:-
#define ENABLE_NOISE_TEXTURE
#define NOISETEXTURE_TILE_SIZE								10.0f
#define NOISETEXTURE_TILES_PER_METRE						(1.0f / NOISETEXTURE_TILE_SIZE)

#endif //#ifndef _3DSMAX_


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Transforms
//
SHARE_PARAM float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
SHARE_PARAM float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;

SHARE_PARAM float4x4 reflectedViewProjTrack
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
>;

SHARE_PARAM float4x4 reflectedViewProjTrackRearView
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
>;

SHARE_PARAM float4x4 reflectedViewProjCars
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
>;

// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;



//-----------------------------------------------------------------------
//
// App parameters
//

float dryingLineWetAmount
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;

float reflectAmountScale
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;

float reflectAmountScaleRearView
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;

float blendEquationBalance
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;

float minDarkeningFactor
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;

float maxDarkeningFactor
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;

float3 darkeningColour
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
>;

float fresnelR0
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;

float minFresnelCalcResult
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
> = 0.0f;




//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// diffuse0
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// diffuse1
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord3 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 3;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// drying line (export of uv's needed)
int texcoord4 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 4;
	int MapChannel = 3;
	int RuntimeTexcoord = 2;
	bool export = false;
> = 0;




#endif

//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture0 : DiffuseMap			// Diffuse colour in RGB
#else
texture diffuseTexture0 : TEXTURE			// Diffuse colour in RGB
#endif
<
	string UIName = "Ch.0 ctrl RGB + A";
	bool appEdit = true;
	bool export = true;
>;

texture diffuseTexture1 : TEXTURE					
<
	string UIName = "Ch.1 ctrl RGB + A";
	bool appEdit = true;
	bool export = true;
>;

texture trackReflectionTexture : TEXTURE					
<
	string UIName = "Track reflection";
	bool appEdit = true;
	bool export = true;
>;

texture trackReflectionTextureRearView : TEXTURE					
<
	string UIName = "Rear view reflection";
	bool appEdit = true;
	bool export = false;
>;

texture carsReflectionTexture : TEXTURE					
<
	string UIName = "Cars reflection";
	bool appEdit = true;
	bool export = true;
>;

texture reflectionNoiseTexture : TEXTURE					
<
	string UIName = "Reflection Noise";
	bool appEdit = true;
	bool export = true;
>;

texture dryingLineTexture : TEXTURE					
<
	string UIName = "Drying Line";
	bool appEdit = true;
	bool export = true;
>;



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap0 : SAMPLER 
< 
    SET_SRGB_TEXTURE
    bool appEdit = false; 
    string SamplerTexture = "diffuseTexture0"; 
#ifdef _3DSMAX_
	string MinFilter = "Linear";
    string MagFilter = "Linear";
#else
    string MinFilter = "Anisotropic";
    string MagFilter = "Anisotropic";
#endif //#ifdef _3DSMAX_
    string MipFilter = "Linear";
    string AddressU  = "Wrap";
    string AddressV  = "Wrap";
>
= sampler_state
{
    Texture = < diffuseTexture0 >;
#if defined(SET_FX_SAMPLER_STATES)
    FX_SAMPLERSTATE_SRGB_TEXTURE
#ifdef _3DSMAX_
	MinFilter = Linear;
    MagFilter = Linear;
#else
	MinFilter = _ANISOMINFILTER;
    MagFilter = _ANISOMAXFILTER;
#endif //#ifdef _3DSMAX_
    MipFilter = Linear;
    AddressU  = Wrap;
    AddressV  = Wrap;
#if defined(_PS3_)
    LODBias = 0;
#else
    MipMapLODBias = 0;
#endif
    SET_MAX_ANISOTROPY( 6 )
#endif
};

sampler2D diffuseMap1 : SAMPLER 
< 
    SET_SRGB_TEXTURE
    bool appEdit = false; 
    string SamplerTexture = "diffuseTexture1";    
#ifdef _3DSMAX_
	string MinFilter = "Linear";
    string MagFilter = "Linear";
#else
    string MinFilter = "Anisotropic";
    string MagFilter = "Anisotropic";
#endif //#ifdef _3DSMAX_
    
    
    string MipFilter = "Linear";
    string AddressU  = "Wrap";
    string AddressV  = "Wrap";
>
= sampler_state
{
    Texture = < diffuseTexture1 >;
#if defined(SET_FX_SAMPLER_STATES)
    FX_SAMPLERSTATE_SRGB_TEXTURE
#ifdef _3DSMAX_
	MinFilter = Linear;
    MagFilter = Linear;
#else
	MinFilter = _ANISOMINFILTER;
    MagFilter = _ANISOMAXFILTER;
#endif //#ifdef _3DSMAX_
    MipFilter = Linear;
    AddressU  = Wrap;
    AddressV  = Wrap;
#if defined(_PS3_)
    LODBias = 0;
#else
    MipMapLODBias = 0;
#endif
    SET_MAX_ANISOTROPY( 6 )
#endif
};


sampler2D trackReflectionMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; string SamplerTexture="trackReflectionTexture"; 
> = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < trackReflectionTexture >;
	MinFilter = _ANISOMINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
	SET_MAX_ANISOTROPY( 6 )
};

sampler2D trackReflectionMapRearView : SAMPLER 
< 	
	SET_SRGB_TEXTURE
	bool appEdit = false; string SamplerTexture="trackReflectionTextureRearView"; 
> = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < trackReflectionTextureRearView >;
	MinFilter = _ANISOMINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
	SET_MAX_ANISOTROPY( 6 )
};

sampler2D carsReflectionMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; string SamplerTexture="carsReflectionTexture"; 
> = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < carsReflectionTexture >;
	MinFilter = _ANISOMINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Clamp;
	SET_MAX_ANISOTROPY( 6 )
};

sampler2D reflectionNoiseMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; string SamplerTexture="reflectionNoiseTexture"; 
> = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < reflectionNoiseTexture >;
	MinFilter = _ANISOMINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
	SET_MAX_ANISOTROPY( 6 )
};

sampler2D dryingLineMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; string SamplerTexture="dryingLineTexture"; 
> = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < dryingLineTexture >;
	MinFilter = _ANISOMINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Clamp;
	AddressV  = Wrap;
	SET_MAX_ANISOTROPY( 6 )
};





//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position					: POSITION;										// Object space position
#ifdef _3DSMAX_
	float3 colour					: TEXCOORD1;									// Vertex colour
	float3 alpha					: TEXCOORD3;									// Vertex alpha
	
	float2 texcoord0				: TEXCOORD0;									// Diffuse map 0
	float2 texcoord1				: TEXCOORD2;									// Diffuse map 1
	float2 dryingLineTexcoord		: TEXCOORD4;									// Drying line calc'ed from AIW
#else
	float4 colour					: COLOR0;										// Vertex colour
	
	float2 texcoord0				: TEXCOORD0;									// Diffuse map 0
	float2 texcoord1				: TEXCOORD1;									// Diffuse map 1
	float2 dryingLineTexcoord		: TEXCOORD2;									// Drying line calc'ed from AIW
#endif //#ifdef _3DSMAX_	
};

// Output structure
struct VSOUTPUT
{
	float4 position					: POSITION;										// View-coords position
	float4 colour					: TEXCOORD5;										// Vertex colour
	float2 texcoord0				: TEXCOORD0;									// Art-created Diffuse / alpha map
	float2 texcoord1				: TEXCOORD1;									// Art-created Diffuse / alpha map
	float2 dryingLineTexcoord		: TEXCOORD2;									// Drying line calc'ed from AIW
	float4 trackReflectTexcoord		: TEXCOORD3;									// Procedural track projection
	float4 carsReflectTexcoord		: TEXCOORD4;									// Procedural cars projection
#ifdef ENABLE_NOISE_TEXTURE
	float2 noiseTexcoord			: TEXCOORD6;									// Procedural noise mapping
#endif //#ifdef ENABLE_NOISE_TEXTURE
	float3 worldEyeVec				: TEXCOORD7;									// eye vec
};

#ifndef _3DSMAX_
// Output structure
struct VSOUTPUTRV
{
	float4 position					: POSITION;										// View-coords position
	float4 colour					: TEXCOORD6;										// Vertex colour
	float2 texcoord0				: TEXCOORD0;									// Art-created Diffuse / alpha map
	float2 texcoord1				: TEXCOORD1;									// Art-created Diffuse / alpha map
	float2 dryingLineTexcoord		: TEXCOORD2;									// Drying line calc'ed from AIW
	float4 trackReflectTexcoord		: TEXCOORD3;									// Procedural track projection
	float2 noiseTexcoord			: TEXCOORD5;									// Procedural noise mapping
};
#endif //#ifndef _3DSMAX_

//-----------------------------------------------------------------------
//
// Vertex shader code
//

//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float CalculateFresnel( float3 _eye, float3 _normal )
{
	// light and normal are assumed to be normalized
	float R0 = fresnelR0;

	return R0 + ( 1.0f - R0 ) * pow( 1.0f - saturate( dot( -_eye, _normal ) ), 5.0f );
}

VSOUTPUT ReflectiveTrackVS( VSINPUT _input )
{
	VSOUTPUT _output;
	
	// TODO: remove / do in MAX?
	_input.position.y +=0.03f;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

#ifdef _3DSMAX_	
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
	_output.texcoord0 = _input.texcoord0;
	_output.texcoord1 = _input.texcoord1;
	_output.dryingLineTexcoord = _input.dryingLineTexcoord;
	_output.trackReflectTexcoord = float4(0.0f, 0.0f, 0.0f, 0.0f);
	_output.carsReflectTexcoord = float4(0.0f, 0.0f, 0.0f, 0.0f);
#ifdef ENABLE_NOISE_TEXTURE
	_output.noiseTexcoord = float2(0.0f, 0.0f);
#endif //#ifdef ENABLE_NOISE_TEXTURE
	_output.worldEyeVec.x = 0.0f;
	_output.worldEyeVec.y = 0.0f;
	_output.worldEyeVec.z = 0.0f;
#else
	
	// Calculate vert's world position
	float4 worldPos = mul( float4( _input.position, 1.0f ), world );
	
	// form uv's for reflection textures
	_output.trackReflectTexcoord = mul( worldPos, reflectedViewProjTrack );
	_output.carsReflectTexcoord = mul( worldPos, reflectedViewProjCars );
	
#ifdef ENABLE_NOISE_TEXTURE
	// noise texcoord
	_output.noiseTexcoord.xy = worldPos.xz * NOISETEXTURE_TILES_PER_METRE;
#endif //#ifdef ENABLE_NOISE_TEXTURE
	
	// pass through drying line + diffuse map etc
	_output.texcoord0 = _input.texcoord0;
	_output.texcoord1 = _input.texcoord1;
	_output.dryingLineTexcoord = _input.dryingLineTexcoord;
	_output.colour = _input.colour;
	
	// eye vec
	_output.worldEyeVec = worldCameraPos - worldPos.xyz;
	
#endif //#ifdef _3DSMAX_	

	return _output;
}

#ifndef _3DSMAX_
VSOUTPUTRV ReflectiveTrackRearViewVS( VSINPUT _input )
{
	VSOUTPUTRV _output;
	
	// TODO: remove / do in MAX?
	_input.position.y +=0.03f;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	
	// Calculate vert's world position
	float4 worldPos = mul( float4( _input.position, 1.0f ), world );
	
	// form uv's for reflection textures
	_output.trackReflectTexcoord = mul( worldPos, reflectedViewProjTrackRearView );
		
	// noise texcoord
#define NOISETEXTURE_TILE_SIZE								10.0f
#define NOISETEXTURE_TILES_PER_METRE						(1.0f / NOISETEXTURE_TILE_SIZE)
	_output.noiseTexcoord.xy = worldPos.xz * NOISETEXTURE_TILES_PER_METRE;
	
	// pass through drying line + diffuse map etc
	_output.texcoord0 = _input.texcoord0;
	_output.texcoord1 = _input.texcoord1;
	_output.dryingLineTexcoord = _input.dryingLineTexcoord;
	_output.colour = _input.colour;
	
	return _output;
}
#endif //#ifndef _3DSMAX_





//-----------------------------------------------------------------------
//
// Pixel Shader(s)
//

// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};

//-----------------------------------------------------------------------
//
// Pixel shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT ReflectiveTrackPS( VSOUTPUT _input )
{
	PSOUTPUT _output;

#ifdef _3DSMAX_
	_output.Colour = tex2D( diffuseMap0, _input.texcoord0 );
	_output.Colour *= tex2D( diffuseMap1, _input.texcoord1 );
	
	_output.Colour.a = 1.0f;
#else

	// combine both diffuse channels (mapped in MAX)
	float4 diffuseX2 = tex2D( diffuseMap0, _input.texcoord0 );
	diffuseX2 *= tex2D( diffuseMap1, _input.texcoord1 );
	
	// read procedurally generated reflection / noise textures
	float3 track = tex2Dproj( trackReflectionMap, _input.trackReflectTexcoord );
	float4 cars = tex2Dproj( carsReflectionMap, _input.carsReflectTexcoord );
#ifdef ENABLE_NOISE_TEXTURE
	float3 noise = tex2D( reflectionNoiseMap, _input.noiseTexcoord );
#endif //#ifdef ENABLE_NOISE_TEXTURE
	float3 dryingLine = tex2D( dryingLineMap, _input.dryingLineTexcoord );
			
	// merge car + track textures
	_output.Colour.rgb = (cars.rgb * cars.a) + (track.rgb * (1.0f - cars.a));
	
	// calculate fresnel, but don't let it fade completely, or road can look dry
	float pixelFresnel = CalculateFresnel( - normalize( _input.worldEyeVec ), float3( 0.0f, 1.0f, 0.0f ));	
	pixelFresnel = max(pixelFresnel, minFresnelCalcResult);	//disabled for now during balancing process of main views (look at later for trackcams)
		
	// calculate the reflection strength
	float reflectAmount;
	reflectAmount = saturate(dryingLine.r + dryingLineWetAmount);
	reflectAmount *= pixelFresnel * reflectAmountScale * _input.colour.a * diffuseX2.a;	
#ifdef ENABLE_NOISE_TEXTURE
	reflectAmount *= noise.r;
#endif //#ifdef ENABLE_NOISE_TEXTURE
			
	// modulate car + track reflection by reflection strength at this pixel
	_output.Colour.rgb *= reflectAmount;
		
	// TODO: if darkening / lightening makes reflections stronger on the light bits, scale refl amount somehow based on 1-darkness?
	// Add on a darkening value - this makes the tarmac darker as it is wet.
	// NOTE: the reason this works is that we use srcBlend = ONE, destBlend = INVSRCALPHA, i.e. premultiplied. Fc = Sc + (1 - As)*Dc
	// NOTE: this relies on a good choice of blendEquationBalance in order that we get the correct alpha blending
	float darkness;
	darkness = lerp(minDarkeningFactor, maxDarkeningFactor, diffuseX2.r);
	_output.Colour.rgb += darkness * darkeningColour;
	_output.Colour.rgb = saturate (_output.Colour.rgb);
				
	// TODO: TO FADE OUT DARKENING ONCE REFLECTION FULLY FADED, FADE blendEquationBalance + DARKENING MAX/MIN TO ZERO.
	// SO, APP WET AMOUNT FIRST SCALES THESE, THEN SCALES ALPHA - FIRST 0.25 = DARK, LAST 0.75 = WET
		
	// set the alpha amount so our blending works correctly
	_output.Colour.a = blendEquationBalance * _input.colour.a;
	
	// fog the reflect mesh to match the rest of the scene (using reconstructed eye vec)
	DO_PS_DISTANCE_FOG( _output.Colour, _input.worldEyeVec );
	
	// fade out the effect as the fog fades in, else we get 'double fog' and the track looks wrong in the distance
	// we scale colour to 0,0,0,0 as that is what results in a unchanged dest pixel with our above blend mode
	// TODO: combine this with the fog calc as part of this is the same calc (or compiler will optimise out?)
	float distSq = dot( _input.worldEyeVec, _input.worldEyeVec );
	float distFadeAmount = saturate( distSq / ( fog_dist_max.x*fog_dist_max.x ) );
	distFadeAmount = 1.0f - distFadeAmount;
	_output.Colour *= distFadeAmount;
	
#ifdef SHOW_DIFFUSE_TEXTURES
	_output.Colour  = diffuseX2;
#endif //#ifdef SHOW_DIFFUSE_TEXTURES

#ifdef SHOW_DARKENING_CALC
	_output.Colour.rgb = darkness * darkeningColour;
	_output.Colour.a = 1.0f;
#endif //#ifdef SHOW_DIFFUSE_TEXTURES

#ifdef SHOW_TRACK_TEXTURE
	_output.Colour.rgb  = track;
	_output.Colour.a = 1.0f;
#endif //#ifdef SHOW_TRACK_TEXTURE

#ifdef SHOW_CARS_TEXTURE
	_output.Colour.rgb = cars;
	_output.Colour.a = 1.0f;
#endif //#ifdef SHOW_CARS_TEXTURE

#ifdef SHOW_FRESNEL
	_output.Colour.r = pixelFresnel;
	_output.Colour.g = pixelFresnel;
	_output.Colour.b = pixelFresnel;
	_output.Colour.a = 1.0f;
#endif //#ifdef SHOW_FRESNEL
	
#endif //#ifdef _3DSMAX_	

	return _output;
}

#ifndef _3DSMAX_
REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT ReflectiveTrackRearViewPS( VSOUTPUTRV _input )
{
	PSOUTPUT _output;

	// combine both diffuse channels (mapped in MAX)
	float4 diffuse = tex2D( diffuseMap0, _input.texcoord0 );
	diffuse *= tex2D( diffuseMap1, _input.texcoord1 );
		
	// read procedurally generated reflection / noise textures
	float3 track = tex2Dproj( trackReflectionMapRearView, _input.trackReflectTexcoord );
	float3 noise = tex2D( reflectionNoiseMap, _input.noiseTexcoord );
	float3 dryingLine = tex2D( dryingLineMap, _input.dryingLineTexcoord );
		
	// track texture only in rear view (in fact just a skydome texture)
	_output.Colour.rgb = track.rgb;
		
	// TODO: less contrast in noise tex?
	// calculate the reflection strength
	float reflectAmount;
	reflectAmount = saturate(dryingLine.r + dryingLineWetAmount);
	reflectAmount *= noise.r * reflectAmountScaleRearView * _input.colour.a * diffuse.a;	
		
	// modulate car + track reflection by reflection strength at this pixel
	_output.Colour.rgb *= reflectAmount;
		
	// TODO: if darkening / lightening makes reflections stronger on the light bits, scale refl amount somehow based on 1-darkness?
	// Add on a darkening value - this makes the tarmac darker as it is wet.
	// NOTE: the reason this works is that we use srcBlend = ONE, destBlend = INVSRCALPHA, i.e. premultiplied. Fc = Sc + (1 - As)*Dc
	// NOTE: this relies on a good choice of blendEquationBalance in order that we get the correct alpha blending
	float darkness;
	darkness = lerp(minDarkeningFactor, maxDarkeningFactor, diffuse.r);
	_output.Colour.rgb += darkness * darkeningColour;
	_output.Colour.rgb = saturate (_output.Colour.rgb);
				
	// TODO: TO FADE OUT DARKENING ONCE REFLECTION FULLY FADED, FADE blendEquationBalance + DARKENING MAX/MIN TO ZERO.
	// SO, APP WET AMOUNT FIRST SCALES THESE, THEN SCALES ALPHA - FIRST 0.25 = DARK, LAST 0.75 = WET
		
	// set the alpha amount so our blending works correctly
	_output.Colour.a = blendEquationBalance * _input.colour.a;
	
	// TODO: copy fogging + fade out calc to here if needed (may not be as rear view draw dist should be low enough so fog is not visible)
	
	return _output;
}
#endif //#ifndef _3DSMAX_


#ifdef _3DSMAX_
PSOUTPUT PSShowDiffuseRGB( VSOUTPUT _input )
{
	PSOUTPUT _output;

	_output.Colour = tex2D( diffuseMap0, _input.texcoord0 );
	_output.Colour *= tex2D( diffuseMap1, _input.texcoord1 );
	
	_output.Colour.a = 1.0f;
	
	return _output;
}

PSOUTPUT PSShowDiffuseAlpha( VSOUTPUT _input )
{
	PSOUTPUT _output;

	_output.Colour = tex2D( diffuseMap0, _input.texcoord0 );
	_output.Colour *= tex2D( diffuseMap1, _input.texcoord1 );
	
	_output.Colour.r = _output.Colour.a;
	_output.Colour.g = _output.Colour.a;
	_output.Colour.b = _output.Colour.a;
	
	_output.Colour.a = 1.0f;
	
	return _output;
}
#endif //#ifdef _3DSMAX_

//-----------------------------------------------------------------------
//
// Technique(s)
//

technique reflectiveTrack
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "reflectiveTrack";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		//string SrcBlend = "SRCALPHA";
		//string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		//SrcBlend = SRCALPHA;
		//DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx ReflectiveTrackVS();
		PixelShader = compile sce_fp_rsx ReflectiveTrackPS();
#else		
		VertexShader = compile vs_3_0 ReflectiveTrackVS();
		PixelShader = compile ps_3_0 ReflectiveTrackPS();
#endif		
	}
}

#ifndef _3DSMAX_
technique reflectiveTrackRearView
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "reflectiveTrackRearView";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
	{

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx ReflectiveTrackRearViewVS();
		PixelShader = compile sce_fp_rsx ReflectiveTrackRearViewPS();
#else		
		VertexShader = compile vs_3_0 ReflectiveTrackRearViewVS();
		PixelShader = compile ps_3_0 ReflectiveTrackRearViewPS();
#endif		
	}
}
#endif //#ifndef _3DSMAX_



//////////////////////////////////////////////////////////////////////////
//
// MAX only techniques to aid artist visualisation
//

#ifdef _3DSMAX_
technique showDiffuseRGB_3DSMAX
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "showDiffuseRGB_3DSMAX";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		//string SrcBlend = "SRCALPHA";
		//string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		//SrcBlend = SRCALPHA;
		//DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

		VertexShader = compile vs_3_0 ReflectiveTrackVS();
		PixelShader = compile ps_3_0 PSShowDiffuseRGB();
	}
}

technique showDiffuseAlpha_3DSMAX
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "showDiffuseAlpha_3DSMAX";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = true;
		string SrcBlend = "ONE";
		string DestBlend = "INVSRCALPHA";
		//string SrcBlend = "SRCALPHA";
		//string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = ONE;
		DestBlend = INVSRCALPHA;
		//SrcBlend = SRCALPHA;
		//DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

		VertexShader = compile vs_3_0 ReflectiveTrackVS();
		PixelShader = compile ps_3_0 PSShowDiffuseAlpha();
	}
}
#endif //#ifdef _3DSMAX_

